(***************************************************************************
 This file is Copyright (C) 2005 Christoph Reichenbach

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the
   Free Software Foundation, Inc.
   59 Temple Place, Suite 330
   Boston, MA  02111-1307
   USA

 The author can be reached as "reichenb" at "colorado.edu".

***************************************************************************)

structure S =
struct
	val print = print
end

structure RLTInt =
struct
	type t = int

	val typename = "integer"
	val print = (S.print o Int.toString)
	val read = Int.fromString
	fun completions () = []
end

structure RLTString =
struct
	type t = string

	val typename = "arbitrary character string"
	val print = S.print
	val read = SOME
	fun completions () = []
end

structure RLTBool =
struct
	type t = bool

	val typename = "truth value"

	fun print true	= S.print "true"
	  | print false	= S.print "false"

	fun read "true"	= SOME true
	  | read "false"= SOME false
	  | read "1"	= SOME true
	  | read "0"	= SOME false
	  | read "t"	= SOME true
	  | read "f"	= SOME false
	  | read "T"	= SOME true
	  | read "F"	= SOME false
	  | read "yes"	= SOME true
	  | read "no"	= SOME false
	  | read _	= NONE

	fun completions () = ["true", "false", "T", "F", "yes", "no"]
end


(* Exported definitions *)


structure RLInt = RLMakeAppResult(structure RLAppType = RLTInt
				  val print = RLTInt.print);
structure RLString = RLMakeAppResult(structure RLAppType = RLTString
				     val print = RLTString.print);
structure RLBool = RLMakeAppResult(structure RLAppType = RLTBool
				   val print = RLTBool.print);

structure RLUnit : READLINE_APP_RESULT =
struct
      type t = unit

      type 'a rlfun_ctx = 'a Readline_Base.rlfun_ctx
      type rlfun = Readline_Base.rlfun

      structure App = struct
		            type t = t
		            val typename = "unit"
		            fun read (s : string) = NONE
		            fun completions () = []
		      end

      structure Result = struct
			       type t = t
			       fun print () = ()
			 end

      (* Hide the argument *)
      fun app (ctx : (unit -> 'a) rlfun_ctx) =
	  { name	= (#name ctx),
	    f		= (fn stringlist =>
			     let val (stringlist_rest, index, f') = (#f ctx) stringlist
			     in (stringlist_rest, index, f' ())
			     end),
            completions = (#completions ctx) }

      fun rlfunc (ctx : unit rlfun_ctx) : rlfun =
	  Readline_Base.new_rlfun (ctx,
				   Result.print) (* Well... not much printing, really *)
end

